#include <Arduino.h>
#include <Wire.h>
#include <Adafruit_GFX.h>
#include <Adafruit_SSD1306.h>
#include <DHT.h>
#include <Adafruit_Sensor.h>

// ---- DISPLAY CONFIG ----
#define SCREEN_WIDTH 128
#define SCREEN_HEIGHT 64
#define OLED_RESET -1
#define OLED_ADDR 0x3C

// ---- PIN CONFIG ----
#define BTN_UP    2
#define BTN_DOWN  3
#define BTN_LEFT  4
#define BTN_RIGHT 5
#define SDA_PIN 6
#define SCL_PIN 7
#define BUZZER_PIN 8
#define DHTPIN 21
#define DHTTYPE DHT11
#define ROT_CLK_PIN  10   // Clock pin
#define ROT_DT_PIN   9    // Data pin
#define ROT_SW_PIN   20   // Switch pin

Adafruit_SSD1306 display(SCREEN_WIDTH, SCREEN_HEIGHT, &Wire, OLED_RESET);
DHT dht(DHTPIN, DHTTYPE);

// ---- Temp & Humidity Data Storage ----
int screenMode = 0;   // 0 = numeric, 1 = temp graph, 2 = humidity graph
float tempHistory[5] = {0, 0, 0, 0, 0};
float humHistory[5] = {0, 0, 0, 0, 0};

// ---- Function to draw a graph with points and numbers ----
void drawGraph(float* data, int len, float yMin, float yMax) {
  int graphWidth = SCREEN_WIDTH - 10;
  int graphHeight = SCREEN_HEIGHT - 15;

  // Draw axes
  display.drawRect(5, 10, graphWidth, graphHeight, SSD1306_WHITE);

  for (int i = 0; i < len-1; i++) {
    // Map oldest on left, newest on right
    int x0 = map(i, 0, len-1, 5, 5+graphWidth);
    int y0 = map(data[i], yMin, yMax, 10+graphHeight, 10);
    int x1 = map(i+1, 0, len-1, 5, 5+graphWidth);
    int y1 = map(data[i+1], yMin, yMax, 10+graphHeight, 10);

    display.drawLine(x0, y0, x1, y1, SSD1306_WHITE);

    // Draw points
    display.fillCircle(x0, y0, 2, SSD1306_WHITE);
    display.fillCircle(x1, y1, 2, SSD1306_WHITE);

    // Draw numbers above points
    display.setTextSize(1);
    display.setCursor(x0 - 4, y0 - 10);
    display.print(data[i], 1);
    display.setCursor(x1 - 4, y1 - 10);
    display.print(data[i+1], 1);
  }
}

void setup() {
  Serial.begin(115200);
  Wire.begin(SDA_PIN, SCL_PIN);

  if (!display.begin(SSD1306_SWITCHCAPVCC, OLED_ADDR)) {
    Serial.println("SSD1306 allocation failed");
    while (true);
  }

  display.clearDisplay();
  display.setTextSize(1);
  display.setTextColor(SSD1306_WHITE);
  display.setCursor(0, 0);
  display.println("Initializing...");
  display.display();

  dht.begin();
  pinMode(BUZZER_PIN, OUTPUT);

  // Buttons are pull-down
  pinMode(BTN_UP, INPUT);
}

void loop() {
  float h = dht.readHumidity();
  float t = dht.readTemperature();

  // Update history arrays (shift left, newest on right)
  for (int i = 0; i < 4; i++) {
    tempHistory[i] = tempHistory[i+1];
    humHistory[i] = humHistory[i+1];
  }
  tempHistory[4] = isnan(t) ? 0 : t;
  humHistory[4] = isnan(h) ? 0 : h;

  // Check BTN_UP press to cycle screens
  if (digitalRead(BTN_UP) == HIGH) {
    screenMode++;
    if (screenMode > 2) screenMode = 0;
    delay(300); // debounce
  }

  // Display screens
  display.clearDisplay();
  switch(screenMode) {
    case 0: // Numeric display
      display.setTextSize(2);
      display.setCursor(0, 0);
      display.print("T: "); display.print(t, 1); display.println("C");
      display.setCursor(0, 30);
      display.print("H: "); display.print(h, 1); display.println("%");
      break;

    case 1: // Temperature graph
      display.setTextSize(1);
      display.setCursor(0,0);
      display.println("Temp graph");
      drawGraph(tempHistory, 5, 0, 50);
      break;

    case 2: // Humidity graph
      display.setTextSize(1);
      display.setCursor(0,0);
      display.println("Humidity graph");
      drawGraph(humHistory, 5, 0, 100);
      break;
  }
  display.display();

  // Buzzer alert if temp > 30°C (short beep pattern)
  if (t > 30.0) {
    tone(BUZZER_PIN, 1000);
    delay(200);
    noTone(BUZZER_PIN);
    delay(800);
  } else {
    noTone(BUZZER_PIN);
    delay(1000);
  }
}