// Snake for ESP32-C3 + Adafruit SSD1306 (128x64) + 4 Pull-Down Buttons + Buzzer
// with Wrap-Around edges
// Libraries: Adafruit_GFX, Adafruit_SSD1306

#include <Arduino.h>
#include <Wire.h>
#include <Adafruit_GFX.h>
#include <Adafruit_SSD1306.h>

// ---- PIN CONFIG ----
#define BTN_UP    2
#define BTN_DOWN  3
#define BTN_LEFT  4
#define BTN_RIGHT 5
#define SDA_PIN 6
#define SCL_PIN 7
#define BUZZER_PIN 8

// ---- DISPLAY CONFIG ----
#define SCREEN_WIDTH 128
#define SCREEN_HEIGHT 64
#define OLED_RESET -1
#define OLED_ADDR 0x3C
Adafruit_SSD1306 display(SCREEN_WIDTH, SCREEN_HEIGHT, &Wire, OLED_RESET);

// ---- SNAKE GAME CONFIG ----
const uint8_t CELL_SIZE = 8;
const uint8_t GRID_W = SCREEN_WIDTH / CELL_SIZE;
const uint8_t GRID_H = SCREEN_HEIGHT / CELL_SIZE;
const uint16_t MAX_SNAKE = GRID_W * GRID_H;

// ---- SNAKE GAME VARIABLES ----
struct Point { uint8_t x, y; };
Point snake[MAX_SNAKE];
uint16_t snakeLen = 3;
int dirX = 1, dirY = 0;
Point food;

uint16_t score = 0;
bool alive = true;

unsigned long lastMoveMillis = 0;
unsigned long moveInterval = 200;

// ---- SNAKE BUTTON HANDLING ----
enum BtnIdx {B_UP=0, B_DOWN=1, B_LEFT=2, B_RIGHT=3};
const uint8_t btnPins[4] = {BTN_UP, BTN_DOWN, BTN_LEFT, BTN_RIGHT};
unsigned long btnLastDebounce[4] = {0,0,0,0};
const unsigned long DEBOUNCE_MS = 60;

// ---- SNAKE BUZZER ----
const int BUZZER_CH = 0;
const int BUZZER_RES = 8;

void beep(int ms=60, int freq=1000) {
  ledcSetup(BUZZER_CH, freq, BUZZER_RES);
  ledcAttachPin(BUZZER_PIN, BUZZER_CH);
  ledcWrite(BUZZER_CH, 200);
  delay(ms);
  ledcWrite(BUZZER_CH, 0);
}

// ---- SNAKE GAME HELPERS ----
long microSeed() {
  long r = 0;
  for (int i=0;i<8;i++) r ^= micros() << (i*3);
  return r;
}

void placeFood() {
  bool used;
  do {
    used = false;
    food.x = random(0, GRID_W);
    food.y = random(0, GRID_H);
    for (uint16_t i=0;i<snakeLen;i++){
      if (snake[i].x == food.x && snake[i].y == food.y) { used = true; break; }
    }
  } while (used);
}

void resetGame() {
  snakeLen = 3;
  snake[0] = {2, (uint8_t)(GRID_H/2)};
  snake[1] = {1, (uint8_t)(GRID_H/2)};
  snake[2] = {0, (uint8_t)(GRID_H/2)};
  dirX = 1; dirY = 0;
  score = 0;
  moveInterval = 220;
  alive = true;
  randomSeed(microSeed());
  placeFood();
}

bool isOpposite(int newDX, int newDY) {
  return (newDX == -dirX && newDY == -dirY);
}

void readButtons() {
  for (int i=0;i<4;i++) {
    int v = digitalRead(btnPins[i]);
    if (v == HIGH) { // pressed (PULL-DOWN)
      unsigned long now = millis();
      if (now - btnLastDebounce[i] > DEBOUNCE_MS) {
        btnLastDebounce[i] = now;
        switch(i) {
          case B_UP:    if (!isOpposite(0,-1))  { dirX=0; dirY=-1; } break;
          case B_DOWN:  if (!isOpposite(0,1))   { dirX=0; dirY=1; }  break;
          case B_LEFT:  if (!isOpposite(-1,0))  { dirX=-1; dirY=0; } break;
          case B_RIGHT: if (!isOpposite(1,0))   { dirX=1; dirY=0; }  break;
        }
      }
    }
  }
}

void moveSnake() {
  Point head = snake[0];
  int nx = head.x + dirX;
  int ny = head.y + dirY;

  // ---- WRAP-AROUND edges ----
  if (nx < 0) nx = GRID_W - 1;
  if (nx >= GRID_W) nx = 0;
  if (ny < 0) ny = GRID_H - 1;
  if (ny >= GRID_H) ny = 0;

  // check self collision
  for (uint16_t i=0;i<snakeLen;i++){
    if (snake[i].x == nx && snake[i].y == ny) {
      alive = false;
      beep(200, 400);
      return;
    }
  }

  for (int i=snakeLen; i>0; i--) {
    snake[i] = snake[i-1];
  }
  snake[0].x = nx;
  snake[0].y = ny;

  if (nx == food.x && ny == food.y) {
    snakeLen++;
    score++;
    beep(50, 1500);
    if (moveInterval > 70) moveInterval -= 8;
    placeFood();
  }
}

void drawGame() {
  display.clearDisplay();

  // snake
  for (uint16_t i=0;i<snakeLen;i++){
    display.fillRect(snake[i].x * CELL_SIZE, snake[i].y * CELL_SIZE, CELL_SIZE, CELL_SIZE, SSD1306_WHITE);
  }

  // food
  display.drawRect(food.x * CELL_SIZE, food.y * CELL_SIZE, CELL_SIZE, CELL_SIZE, SSD1306_WHITE);

  // score
  display.setTextSize(1);
  display.setTextColor(SSD1306_WHITE);
  display.setCursor(2, 2);
  display.print("Score:");
  display.print(score);

  if (!alive) {
    display.fillRect(12, 18, SCREEN_WIDTH-24, 28, SSD1306_WHITE);
    display.setTextColor(SSD1306_BLACK);
    display.setCursor(22, 26);
    display.setTextSize(2);
    display.print("GAME");
    display.setCursor(22, 44);
    display.print("OVER");
    display.setTextSize(1);
  }

  display.display();
}

void setupPins() {
  pinMode(BUZZER_PIN, OUTPUT);
  for (int i=0;i<4;i++){
    pinMode(btnPins[i], INPUT); // pull-down buttons
  }
}

void setup() {
  Serial.begin(115200);
  delay(10);

  Wire.begin(SDA_PIN, SCL_PIN);
  if(!display.begin(SSD1306_SWITCHCAPVCC, OLED_ADDR)) {
    Serial.println(F("SSD1306 allocation failed"));
    for(;;);
  }
  display.clearDisplay();
  display.display();

  setupPins();
  beep(40, 1200);

  resetGame();
  lastMoveMillis = millis();

  Serial.println("Snake game started (wrap-around enabled).");
}

void loop() {
  readButtons();

  if (!alive) {
    for (int i=0;i<4;i++) {
      if (digitalRead(btnPins[i]) == HIGH) { // pressed
        resetGame();
        delay(200);
        lastMoveMillis = millis();
        return;
      }
    }
    drawGame();
    delay(60);
    return;
  }

  unsigned long now = millis();
  if (now - lastMoveMillis >= moveInterval) {
    lastMoveMillis = now;
    moveSnake();
    drawGame();
  }
}
