#include <Arduino.h>
#include <Wire.h>
#include <Adafruit_GFX.h>
#include <Adafruit_SSD1306.h>
#include <DHT.h>
#include <Adafruit_Sensor.h>

// ---- DISPLAY CONFIG ----
#define SCREEN_WIDTH 128
#define SCREEN_HEIGHT 64
#define OLED_RESET -1
#define OLED_ADDR 0x3C

// ---- PIN CONFIG ----
#define BTN_LEFT  4   // move UP
#define BTN_RIGHT 5   // move DOWN
#define SDA_PIN 6
#define SCL_PIN 7
#define BUZZER_PIN 8
#define DHTPIN 21
#define DHTTYPE DHT11

Adafruit_SSD1306 display(SCREEN_WIDTH, SCREEN_HEIGHT, &Wire, OLED_RESET);

// Pong Variables
// ---- Pong Player Paddle ----
int paddleY = 20;
int paddleHeight = 16;
int paddleWidth = 3;
// ---- AI Paddle ----
int aiY = 20;
int aiSpeed = 2;        // difficulty: increase for harder AI
int aiHeight = 16;
int aiWidth = 3;
// ---- Ball ----
float ballX = 60;
float ballY = 32;
float ballSpeedX = -2;
float ballSpeedY = 1.5;
// ---- Frame ----
unsigned long ponglastFrame = 0;
int pongfps = 30;
// ---- Score ----
int pongscore = 0;

// Pong Functions
void pongbeep(int duration = 40) {
  digitalWrite(BUZZER_PIN, HIGH);
  delay(duration);
  digitalWrite(BUZZER_PIN, LOW);
}

void resetBall() {
  ballX = SCREEN_WIDTH / 2;
  ballY = SCREEN_HEIGHT / 2;
  ballSpeedX = (random(0, 2) ? -2 : 2);
  ballSpeedY = (random(0, 2) ? -1.5 : 1.5);
  pongbeep(150);
}

void setup() {
  pinMode(BTN_LEFT, INPUT_PULLDOWN);
  pinMode(BTN_RIGHT, INPUT_PULLDOWN);
  pinMode(BUZZER_PIN, OUTPUT);

  Wire.begin(SDA_PIN, SCL_PIN);
  display.begin(SSD1306_SWITCHCAPVCC, OLED_ADDR);
  display.clearDisplay();
  display.display();

  randomSeed(analogRead(0));
}

void loop() {
  unsigned long now = millis();
  if (now - ponglastFrame < 1000 / pongfps) return;
  ponglastFrame = now;

  // --- Player Paddle Movement ---
  if (digitalRead(BTN_LEFT)) {
    paddleY -= 3;
    if (paddleY < 0) paddleY = 0;
  }
  if (digitalRead(BTN_RIGHT)) {
    paddleY += 3;
    if (paddleY + paddleHeight > SCREEN_HEIGHT)
      paddleY = SCREEN_HEIGHT - paddleHeight;
  }

  // --- AI Paddle Movement ---
  if (ballY > aiY + aiHeight / 2) aiY += aiSpeed;
  else if (ballY < aiY + aiHeight / 2) aiY -= aiSpeed;

  // Clamp AI paddle
  if (aiY < 0) aiY = 0;
  if (aiY + aiHeight > SCREEN_HEIGHT) aiY = SCREEN_HEIGHT - aiHeight;

  // --- Ball Movement ---
  ballX += ballSpeedX;
  ballY += ballSpeedY;

  // Bounce top & bottom
  if (ballY <= 0 || ballY >= SCREEN_HEIGHT - 2) {
    ballSpeedY = -ballSpeedY;
    pongbeep(30);
  }

  // Player paddle collision
  if (ballX <= paddleWidth + 2) {
    if (ballY >= paddleY && ballY <= paddleY + paddleHeight) {
      ballSpeedX = -ballSpeedX;
      pongscore++;
      pongbeep(80);
    }
  }

  // AI paddle collision
  if (ballX >= SCREEN_WIDTH - aiWidth - 3) {
    if (ballY >= aiY && ballY <= aiY + aiHeight) {
      ballSpeedX = -ballSpeedX;
      pongbeep(60);
    }
  }

  // Miss → score reset
  if (ballX < 0) {
    pongscore = 0;
    resetBall();
  }

  // ---- DRAW EVERYTHING ----
  display.clearDisplay();

  // Player paddle
  display.fillRect(0, paddleY, paddleWidth, paddleHeight, SSD1306_WHITE);

  // AI paddle
  display.fillRect(SCREEN_WIDTH - aiWidth, aiY, aiWidth, aiHeight, SSD1306_WHITE);

  // Ball
  display.fillRect(ballX, ballY, 3, 3, SSD1306_WHITE);

  // Mid dashed line
  for (int i = 0; i < SCREEN_HEIGHT; i += 6)
    display.drawFastVLine(SCREEN_WIDTH / 2, i, 3, SSD1306_WHITE);

  // Score
  display.setTextSize(1);
  display.setTextColor(SSD1306_WHITE);
  display.setCursor(90, 0);
  display.print("S:");
  display.print(pongscore);

  display.display();
}
