#include <Arduino.h>
#include <Wire.h>
#include <Adafruit_GFX.h>
#include <Adafruit_SSD1306.h>
#include <DHT.h>
#include <Adafruit_Sensor.h>

// ---- DISPLAY CONFIG ----
#define SCREEN_WIDTH 128
#define SCREEN_HEIGHT 64
#define OLED_RESET -1
#define OLED_ADDR 0x3C
Adafruit_SSD1306 display(SCREEN_WIDTH, SCREEN_HEIGHT, &Wire, OLED_RESET);

// ---- PIN CONFIG ----
#define BTN_UP    2
#define BTN_DOWN  3
#define BTN_LEFT  4
#define BTN_RIGHT 5
#define SDA_PIN   6
#define SCL_PIN   7
#define BUZZER_PIN 8
#define ROT_DT_PIN   9
#define ROT_CLK_PIN  10
#define ROT_SW_PIN   20
#define DHTPIN 21
#define DHTTYPE DHT11
DHT dht(DHTPIN, DHTTYPE);

// ---- MENU DATA ----
const char* menuItems[] = {"Buzzer Test", "Temp & Humidity", "Snake Game"};
int menuLength = 3;
int selectedIndex = 0;
bool inMenuItem = false;

// Rotary encoder state
int lastCLK = HIGH;

// Non-blocking debounce
unsigned long lastEncoderTime = 0;
const unsigned long ENCODER_DEBOUNCE = 75;  // ms

unsigned long lastBtnUpTime = 0;
const unsigned long BUTTON_DEBOUNCE = 150;  // ms


// --------------------------------------
// DISPLAY MENU
//---------------------------------------
void displayMenu() {
  display.clearDisplay();
  display.setCursor(0, 0);

  if (!inMenuItem) {
    display.println("== MENU ==");

    for (int i = 0; i < menuLength; i++) {
      if (i == selectedIndex) display.print("> ");
      else display.print("  ");
      display.println(menuItems[i]);
    }

  } else {
    display.println(menuItems[selectedIndex]);
    display.println("ITEM ACTIVE");
    display.println("Rotate encoder");
    display.println("to EXIT");
  }

  display.display();
}


// --------------------------------------
// READ ENCODER (scroll menu or exit item)
//---------------------------------------
void readEncoder() {
  int currentCLK = digitalRead(ROT_CLK_PIN);
  unsigned long now = millis();

  if (currentCLK != lastCLK && (now - lastEncoderTime) > ENCODER_DEBOUNCE) {
    lastEncoderTime = now;

    // Exit item if inside
    if (inMenuItem) {
      inMenuItem = false;
      lastCLK = currentCLK; // reset encoder state
      return;
    }

    // Normal menu scrolling
    if (digitalRead(ROT_DT_PIN) != currentCLK) {
      selectedIndex++;
      if (selectedIndex >= menuLength) selectedIndex = 0;
    } else {
      selectedIndex--;
      if (selectedIndex < 0) selectedIndex = menuLength - 1;
    }
  }

  lastCLK = currentCLK;
}


// --------------------------------------
// SETUP
//---------------------------------------
void setup() {
  pinMode(ROT_CLK_PIN, INPUT_PULLUP);
  pinMode(ROT_DT_PIN, INPUT_PULLUP);
  pinMode(ROT_SW_PIN, INPUT_PULLUP);

  // Buttons are pull-down
  pinMode(BTN_UP, INPUT);
  pinMode(BTN_DOWN, INPUT);
  pinMode(BTN_LEFT, INPUT);
  pinMode(BTN_RIGHT, INPUT);

  pinMode(BUZZER_PIN, OUTPUT);

  Serial.begin(115200);

  Wire.begin(SDA_PIN, SCL_PIN);
  display.begin(SSD1306_SWITCHCAPVCC, OLED_ADDR);
  display.setTextSize(1);
  display.setTextColor(SSD1306_WHITE);

  dht.begin();

  lastCLK = digitalRead(ROT_CLK_PIN);
}


// --------------------------------------
// LOOP
//---------------------------------------
void loop() {
  readEncoder();
  displayMenu();

  // ENTER ITEM with non-blocking debounce
  if (digitalRead(BTN_UP) == HIGH && !inMenuItem) {
    unsigned long now = millis();
    if ((now - lastBtnUpTime) > BUTTON_DEBOUNCE) {
      inMenuItem = true;
      lastBtnUpTime = now;
    }
  }
}
