#include <Arduino.h>
#include <Wire.h>
#include <Adafruit_GFX.h>
#include <Adafruit_SSD1306.h>
#include <DHT.h>
#include <Adafruit_Sensor.h>

// ---- DISPLAY CONFIG ----
#define SCREEN_WIDTH 128
#define SCREEN_HEIGHT 64
#define OLED_RESET -1
#define OLED_ADDR 0x3C
Adafruit_SSD1306 display(SCREEN_WIDTH, SCREEN_HEIGHT, &Wire, OLED_RESET);

// ---- PIN CONFIG ----
#define BTN_UP    2
#define BTN_DOWN  3
#define BTN_LEFT  4
#define BTN_RIGHT 5
#define SDA_PIN   6
#define SCL_PIN   7
#define BUZZER_PIN 8
#define ROT_DT_PIN   9
#define ROT_CLK_PIN  10
#define ROT_SW_PIN   20
#define DHTPIN 21
#define DHTTYPE DHT11
DHT dht(DHTPIN, DHTTYPE);

// ---- MENU DATA ----
const char* menuItems[] = {"Buzzer Test", "Temp & Humidity", "Snake Game", "Flappy Bird", "Pong Game"};
int menuLength = 5;
int selectedIndex = 0;
bool inMenuItem = false;

// Rotary encoder state
int lastCLK = HIGH;

// Non-blocking debounce
unsigned long lastEncoderTime = 0;
const unsigned long ENCODER_DEBOUNCE = 75;  // ms

unsigned long lastBtnUpTime = 0;
const unsigned long BUTTON_DEBOUNCE = 150;  // ms



// buzzer test frequencies
u_int32_t Frequencies[] = {400,800,1600,3200,6400,10000,20000};
uint32_t chosen_Freq = 0;
bool Started = false;

// buzzer test display
void DisplayRenderer() {
  display.clearDisplay();
  display.setTextSize(1);
  display.setTextColor(WHITE);
  display.setCursor(43, 0);
  display.println(String(Frequencies[chosen_Freq])+"Hz");
  display.setCursor(16, 23);
  display.println("+");
  display.setCursor(50, 23);
  if (Started) {
    display.println("Stop");
  }
  else {
    display.println("Start");
  }
  display.setCursor(97, 23);
  display.println("-");
  display.display();
}



// ---- Temp & Humidity Data Storage ----
int screenMode = 0;   // 0 = numeric, 1 = temp graph, 2 = humidity graph
float tempHistory[5] = {0, 0, 0, 0, 0};
float humHistory[5] = {0, 0, 0, 0, 0};
float h = 0;
float t = 0;

// ---- Function to draw a graph with points and numbers ----
void drawGraph(float* data, int len, float yMin, float yMax) {
  int graphWidth = SCREEN_WIDTH - 10;
  int graphHeight = SCREEN_HEIGHT - 15;

  // Draw axes
  display.drawRect(5, 10, graphWidth, graphHeight, SSD1306_WHITE);

  for (int i = 0; i < len-1; i++) {
    // Map oldest on left, newest on right
    int x0 = map(i, 0, len-1, 5, 5+graphWidth);
    int y0 = map(data[i], yMin, yMax, 10+graphHeight, 10);
    int x1 = map(i+1, 0, len-1, 5, 5+graphWidth);
    int y1 = map(data[i+1], yMin, yMax, 10+graphHeight, 10);

    display.drawLine(x0, y0, x1, y1, SSD1306_WHITE);

    // Draw points
    display.fillCircle(x0, y0, 2, SSD1306_WHITE);
    display.fillCircle(x1, y1, 2, SSD1306_WHITE);

    // Draw numbers above points
    display.setTextSize(1);
    display.setCursor(x0 - 4, y0 - 10);
    display.print(data[i], 1);
    display.setCursor(x1 - 4, y1 - 10);
    display.print(data[i+1], 1);
  }
}



// ---- SNAKE GAME CONFIG ----
const uint8_t CELL_SIZE = 8;
const uint8_t GRID_W = SCREEN_WIDTH / CELL_SIZE;
const uint8_t GRID_H = SCREEN_HEIGHT / CELL_SIZE;
const uint16_t MAX_SNAKE = GRID_W * GRID_H;

// ---- SNAKE GAME VARIABLES ----
struct Point { uint8_t x, y; };
Point snake[MAX_SNAKE];
uint16_t snakeLen = 3;
int dirX = 1, dirY = 0;
Point food;

uint16_t score = 0;
bool alive = true;

unsigned long lastMoveMillis = 0;
unsigned long moveInterval = 200;
unsigned long now = 0;

// ---- SNAKE BUTTON HANDLING ----
enum BtnIdx {B_UP=0, B_DOWN=1, B_LEFT=2, B_RIGHT=3};
const uint8_t btnPins[4] = {BTN_UP, BTN_DOWN, BTN_LEFT, BTN_RIGHT};
unsigned long btnLastDebounce[4] = {0,0,0,0};
const unsigned long DEBOUNCE_MS = 60;

// ---- SNAKE BUZZER ----
const int BUZZER_CH = 0;
const int BUZZER_RES = 8;

void beep(int ms=60, int freq=1000) {
  ledcSetup(BUZZER_CH, freq, BUZZER_RES);
  ledcAttachPin(BUZZER_PIN, BUZZER_CH);
  ledcWrite(BUZZER_CH, 200);
  delay(ms);
  ledcWrite(BUZZER_CH, 0);
}

// ---- SNAKE GAME HELPERS ----
long microSeed() {
  long r = 0;
  for (int i=0;i<8;i++) r ^= micros() << (i*3);
  return r;
}

void placeFood() {
  bool used;
  do {
    used = false;
    food.x = random(0, GRID_W);
    food.y = random(0, GRID_H);
    for (uint16_t i=0;i<snakeLen;i++){
      if (snake[i].x == food.x && snake[i].y == food.y) { used = true; break; }
    }
  } while (used);
}

void resetGame() {
  snakeLen = 3;
  snake[0] = {2, (uint8_t)(GRID_H/2)};
  snake[1] = {1, (uint8_t)(GRID_H/2)};
  snake[2] = {0, (uint8_t)(GRID_H/2)};
  dirX = 1; dirY = 0;
  score = 0;
  moveInterval = 220;
  alive = true;
  randomSeed(microSeed());
  placeFood();
}

bool isOpposite(int newDX, int newDY) {
  return (newDX == -dirX && newDY == -dirY);
}

void readButtons() {
  for (int i=0;i<4;i++) {
    int v = digitalRead(btnPins[i]);
    if (v == HIGH) { // pressed (PULL-DOWN)
      unsigned long now = millis();
      if (now - btnLastDebounce[i] > DEBOUNCE_MS) {
        btnLastDebounce[i] = now;
        switch(i) {
          case B_UP:    if (!isOpposite(0,-1))  { dirX=0; dirY=-1; } break;
          case B_DOWN:  if (!isOpposite(0,1))   { dirX=0; dirY=1; }  break;
          case B_LEFT:  if (!isOpposite(-1,0))  { dirX=-1; dirY=0; } break;
          case B_RIGHT: if (!isOpposite(1,0))   { dirX=1; dirY=0; }  break;
        }
      }
    }
  }
}

void moveSnake() {
  Point head = snake[0];
  int nx = head.x + dirX;
  int ny = head.y + dirY;

  // ---- WRAP-AROUND edges ----
  if (nx < 0) nx = GRID_W - 1;
  if (nx >= GRID_W) nx = 0;
  if (ny < 0) ny = GRID_H - 1;
  if (ny >= GRID_H) ny = 0;

  // check self collision
  for (uint16_t i=0;i<snakeLen;i++){
    if (snake[i].x == nx && snake[i].y == ny) {
      alive = false;
      beep(200, 400);
      return;
    }
  }

  for (int i=snakeLen; i>0; i--) {
    snake[i] = snake[i-1];
  }
  snake[0].x = nx;
  snake[0].y = ny;

  if (nx == food.x && ny == food.y) {
    snakeLen++;
    score++;
    beep(50, 1500);
    if (moveInterval > 70) moveInterval -= 8;
    placeFood();
  }
}

void drawGame() {
  display.clearDisplay();

  // snake
  for (uint16_t i=0;i<snakeLen;i++){
    display.fillRect(snake[i].x * CELL_SIZE, snake[i].y * CELL_SIZE, CELL_SIZE, CELL_SIZE, SSD1306_WHITE);
  }

  // food
  display.drawRect(food.x * CELL_SIZE, food.y * CELL_SIZE, CELL_SIZE, CELL_SIZE, SSD1306_WHITE);

  // score
  display.setTextSize(1);
  display.setTextColor(SSD1306_WHITE);
  display.setCursor(2, 2);
  display.print("Score:");
  display.print(score);

  if (!alive) {
    display.fillRect(12, 18, SCREEN_WIDTH-24, 28, SSD1306_WHITE);
    display.setTextColor(SSD1306_BLACK);
    display.setCursor(22, 26);
    display.setTextSize(2);
    display.print("GAME");
    display.setCursor(22, 44);
    display.print("OVER");
    display.setTextSize(1);
  }

  display.display();
}



// ---- Flappy Bird Game Variables ----
float birdY = 32;
float birdVelocity = 0;
float gravity = 0.4;
float flapStrength = -2.8;

int pipeX = SCREEN_WIDTH;
int pipeGap = 25;
int pipeTopHeight;

bool flappygameOver = false;

// ---- Flappy Bird Functions ----
void flappyresetGame() {
  birdY = SCREEN_HEIGHT / 2;
  birdVelocity = 0;
  pipeX = SCREEN_WIDTH;
  pipeTopHeight = random(10, SCREEN_HEIGHT - pipeGap - 10);
  flappygameOver = false;
}

void flappybeep(short freq, short dur) {
  tone(BUZZER_PIN, freq);
  delay(dur);
  noTone(BUZZER_PIN);
}



// switch to run the current menu item selected
void runCurrentItem(){
  switch (selectedIndex)
  {
  case 0:  // buzzer test
    if (digitalRead(BTN_UP) == HIGH) {
      if (Started) {
        noTone(BUZZER_PIN);
        Started = false;
      }
      else {
        tone(BUZZER_PIN, Frequencies[chosen_Freq]);
        Started = true;
      }
      DisplayRenderer();
      delay(150);
    }
    if (digitalRead(BTN_LEFT) == HIGH) {
      if (chosen_Freq < 6) {
        chosen_Freq++;
        if (Started) {
          tone(BUZZER_PIN, Frequencies[chosen_Freq]);
        }
      }
      DisplayRenderer();
      delay(150);
    }
    if (digitalRead(BTN_RIGHT) == HIGH) {
      if (chosen_Freq > 0 ) {
        chosen_Freq--;
        if (Started) {
          tone(BUZZER_PIN, Frequencies[chosen_Freq]);
        }
      }
      DisplayRenderer();
      delay(150);
    }
    break;



  case 1: // temp & humidity sensor
    t = tempHistory[4];
    h = humHistory[4];

    // Check BTN_UP press to cycle screens
    if (digitalRead(BTN_UP) == HIGH) {
      screenMode++;
      if (screenMode > 2) screenMode = 0;
      delay(300); // debounce
    }

    // Display screens
    display.clearDisplay();
    switch(screenMode) {
      case 0: // Numeric display
        display.setTextSize(2);
        display.setCursor(0, 0);
        display.print("T: "); display.print(t, 1); display.println("C");
        display.setCursor(0, 30);
        display.print("H: "); display.print(h, 1); display.println("%");
        break;

      case 1: // Temperature graph
        display.setTextSize(1);
        display.setCursor(0,0);
        display.println("Temp graph");
        drawGraph(tempHistory, 5, 0, 50);
        break;

      case 2: // Humidity graph
        display.setTextSize(1);
        display.setCursor(0,0);
        display.println("Humidity graph");
        drawGraph(humHistory, 5, 0, 100);
        break;
    }
    display.display();

    // Buzzer alert if temp > 30°C (short beep pattern)
    if (t > 30.0) {
      tone(BUZZER_PIN, 1000);
      delay(200);
      noTone(BUZZER_PIN);
      delay(800);
    } 
    else {
      noTone(BUZZER_PIN);
      delay(1000);
    }
    break;


    
  case 2: // snake game
    readButtons();

    if (!alive) {
      for (int i=0;i<4;i++) {
        if (digitalRead(btnPins[i]) == HIGH) { // pressed
          resetGame();
          delay(200);
          lastMoveMillis = millis();
          return;
        }
      }
      drawGame();
      delay(60);
      return;
    }

    now = millis();
    if (now - lastMoveMillis >= moveInterval) {
      lastMoveMillis = now;
      moveSnake();
      drawGame();
    }
    break;
  

  
  case 3: // flappy bird
    if (!flappygameOver) {
      // -------- Bird Physics --------
      birdVelocity += gravity;
      birdY += birdVelocity;

      // -------- Pipe Movement --------
      pipeX -= 2;
      if (pipeX < -20) {
        pipeX = SCREEN_WIDTH;
        pipeTopHeight = random(10, SCREEN_HEIGHT - pipeGap - 10);
      }

      // -------- Button: Flap --------
      if (digitalRead(BTN_UP) == HIGH) {
        birdVelocity = flapStrength;
        flappybeep(1200, 60);     // jump sound
      }

      // -------- Collision Check --------
      bool hitGround = (birdY > SCREEN_HEIGHT - 3);
      bool hitCeiling = (birdY < 0);

      bool hitPipe =
        (pipeX < 20 && pipeX > 0) && 
        (birdY < pipeTopHeight || birdY > pipeTopHeight + pipeGap);

      if (hitGround || hitCeiling || hitPipe) {
        flappygameOver = true;
        flappybeep(200, 300);   // crash sound
      }

      // -------- Draw Frame --------
      display.clearDisplay();

      // Bird
      display.fillCircle(15, birdY, 3, SSD1306_WHITE);

      // Pipe top
      display.fillRect(pipeX, 0, 20, pipeTopHeight, SSD1306_WHITE);

      // Pipe bottom
      display.fillRect(pipeX, pipeTopHeight + pipeGap, 20, SCREEN_HEIGHT, SSD1306_WHITE);

      display.display();
      delay(30);
    }
    else {
      // -------- Game Over Screen --------
      display.clearDisplay();
      display.setTextSize(2);
      display.setCursor(15, 10);
      display.println("GAME");
      display.setCursor(15, 35);
      display.println("OVER");
      display.display();

      // Press UP to restart
      if (digitalRead(BTN_UP) == HIGH) {
        delay(200);
        flappyresetGame();
      }
    }
    break;


  case 4: // pong game
    break;



  default:
    break;
  }
}

// --------------------------------------
// DISPLAY MENU
//---------------------------------------
void displayMenu() {
  display.clearDisplay();
  display.setCursor(0, 0);

  if (!inMenuItem) {
    display.setTextSize(1);
    display.println("== MENU ==");

    for (int i = 0; i < menuLength; i++) {
      if (i == selectedIndex) display.print("> ");
      else display.print("  ");
      display.println(menuItems[i]);
    }

  } else {
    /*
    display.println(menuItems[selectedIndex]);
    display.println("ITEM ACTIVE");
    display.println("Rotate encoder");
    display.println("to EXIT");
    */
  }

  display.display();
}

// --------------------------------------
// READ ENCODER (scroll menu or exit item)
//---------------------------------------
void readEncoder() {
  int currentCLK = digitalRead(ROT_CLK_PIN);
  unsigned long now = millis();

  if (currentCLK != lastCLK && (now - lastEncoderTime) > ENCODER_DEBOUNCE) {
    lastEncoderTime = now;

    // Exit item if inside
    if (inMenuItem) {
      inMenuItem = false;
      lastCLK = currentCLK; // reset encoder state
      return;
    }

    // Normal menu scrolling
    if (digitalRead(ROT_DT_PIN) != currentCLK) {
      selectedIndex++;
      if (selectedIndex >= menuLength) selectedIndex = 0;
    } else {
      selectedIndex--;
      if (selectedIndex < 0) selectedIndex = menuLength - 1;
    }
  }

  lastCLK = currentCLK;
}


// --------------------------------------
// SETUP
//---------------------------------------
void setup() {
  pinMode(ROT_CLK_PIN, INPUT_PULLUP);
  pinMode(ROT_DT_PIN, INPUT_PULLUP);
  pinMode(ROT_SW_PIN, INPUT_PULLUP);

  // Buttons are pull-down
  pinMode(BTN_UP, INPUT);
  pinMode(BTN_DOWN, INPUT);
  pinMode(BTN_LEFT, INPUT);
  pinMode(BTN_RIGHT, INPUT);

  pinMode(BUZZER_PIN, OUTPUT);

  Serial.begin(115200);

  Wire.begin(SDA_PIN, SCL_PIN);
  display.begin(SSD1306_SWITCHCAPVCC, OLED_ADDR);
  display.setTextSize(1);
  display.setTextColor(SSD1306_WHITE);

  dht.begin();

  lastCLK = digitalRead(ROT_CLK_PIN);
}


// --------------------------------------
// LOOP
//---------------------------------------
void loop() {
  readEncoder();
  
  // temp and humidity code
  float h = dht.readHumidity();
  float t = dht.readTemperature();

  // Update history arrays (shift left, newest on right)
  for (int i = 0; i < 4; i++) {
    tempHistory[i] = tempHistory[i+1];
    humHistory[i] = humHistory[i+1];
  }
  tempHistory[4] = isnan(t) ? 0 : t;
  humHistory[4] = isnan(h) ? 0 : h;

  // check whether inside a menu item
  if (inMenuItem)
  { runCurrentItem(); }
  else
  { displayMenu(); }

  // ENTER ITEM with non-blocking debounce
  if (digitalRead(BTN_UP) == HIGH && !inMenuItem) {
    unsigned long now = millis();
    if ((now - lastBtnUpTime) > BUTTON_DEBOUNCE) {
      inMenuItem = true;
      lastBtnUpTime = now;
    }
  }
}
