#include <Arduino.h>
#include <Wire.h>
#include <Adafruit_GFX.h>
#include <Adafruit_SSD1306.h>

// ---- DISPLAY CONFIG ----
#define SCREEN_WIDTH 128
#define SCREEN_HEIGHT 64
#define OLED_RESET -1
#define OLED_ADDR 0x3C

// ---- PIN CONFIG ----
#define BTN_UP    2
#define BTN_DOWN  3
#define BTN_LEFT  4
#define BTN_RIGHT 5
#define SDA_PIN 6
#define SCL_PIN 7
#define BUZZER_PIN 8
#define ROT_DT_PIN   9
#define ROT_CLK_PIN  10
#define ROT_SW_PIN   20 
#define DHTPIN 21
#define DHTTYPE DHT11

Adafruit_SSD1306 display(SCREEN_WIDTH, SCREEN_HEIGHT, &Wire, OLED_RESET);

// ---- Flappy Bird Game Variables ----
float birdY = 32;
float birdVelocity = 0;
float gravity = 0.4;
float flapStrength = -2.8;

int pipeX = SCREEN_WIDTH;
int pipeGap = 25;
int pipeTopHeight;

bool flappygameOver = false;

// ---- Flappy Bird Functions ----
void flappyresetGame() {
  birdY = SCREEN_HEIGHT / 2;
  birdVelocity = 0;
  pipeX = SCREEN_WIDTH;
  pipeTopHeight = random(10, SCREEN_HEIGHT - pipeGap - 10);
  flappygameOver = false;
}

void flappybeep(short freq, short dur) {
  tone(BUZZER_PIN, freq);
  delay(dur);
  noTone(BUZZER_PIN);
}

void setup() {
  Serial.begin(115200);

  Wire.begin(SDA_PIN, SCL_PIN);

  if (!display.begin(SSD1306_SWITCHCAPVCC, OLED_ADDR)) {
    Serial.println("SSD1306 allocation failed!");
    while (true);
  }

  display.clearDisplay();
  display.setTextSize(1);
  display.setTextColor(SSD1306_WHITE);
  display.setCursor(0, 0);
  display.println("Flappy Bird Loading...");
  display.display();

  pinMode(BTN_UP, INPUT);     // pull-down button
  pinMode(BUZZER_PIN, OUTPUT);

  randomSeed(analogRead(0));

  flappyresetGame();
}

void loop() {
  if (!flappygameOver) {
    // -------- Bird Physics --------
    birdVelocity += gravity;
    birdY += birdVelocity;

    // -------- Pipe Movement --------
    pipeX -= 2;
    if (pipeX < -20) {
      pipeX = SCREEN_WIDTH;
      pipeTopHeight = random(10, SCREEN_HEIGHT - pipeGap - 10);
    }

    // -------- Button: Flap --------
    if (digitalRead(BTN_UP) == HIGH) {
      birdVelocity = flapStrength;
      flappybeep(1200, 60);     // jump sound
    }

    // -------- Collision Check --------
    bool hitGround = (birdY > SCREEN_HEIGHT - 3);
    bool hitCeiling = (birdY < 0);

    bool hitPipe =
      (pipeX < 20 && pipeX > 0) && 
      (birdY < pipeTopHeight || birdY > pipeTopHeight + pipeGap);

    if (hitGround || hitCeiling || hitPipe) {
      flappygameOver = true;
      flappybeep(200, 300);   // crash sound
    }

    // -------- Draw Frame --------
    display.clearDisplay();

    // Bird
    display.fillCircle(15, birdY, 3, SSD1306_WHITE);

    // Pipe top
    display.fillRect(pipeX, 0, 20, pipeTopHeight, SSD1306_WHITE);

    // Pipe bottom
    display.fillRect(pipeX, pipeTopHeight + pipeGap, 20, SCREEN_HEIGHT, SSD1306_WHITE);

    display.display();
    delay(30);
  }
  else {
    // -------- Game Over Screen --------
    display.clearDisplay();
    display.setTextSize(2);
    display.setCursor(15, 10);
    display.println("GAME");
    display.setCursor(15, 35);
    display.println("OVER");
    display.display();

    // Press UP to restart
    if (digitalRead(BTN_UP) == HIGH) {
      delay(200);
      flappyresetGame();
    }
  }
}
